// VidSnap Chrome Extension - Content Script
// This script runs on Facebook pages to add download buttons to videos

(function() {
  'use strict';
  
  const VIDSNAP_URL = 'https://vidsnap.net';
  let processedVideos = new Set();
  
  // Create download button element
  function createDownloadButton(videoUrl) {
    const btn = document.createElement('button');
    btn.className = 'vidsnap-download-btn';
    btn.innerHTML = `
      <svg width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
        <path d="M.5 9.9a.5.5 0 0 1 .5.5v2.5a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-2.5a.5.5 0 0 1 1 0v2.5a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2v-2.5a.5.5 0 0 1 .5-.5z"/>
        <path d="M7.646 11.854a.5.5 0 0 0 .708 0l3-3a.5.5 0 0 0-.708-.708L8.5 10.293V1.5a.5.5 0 0 0-1 0v8.793L5.354 8.146a.5.5 0 1 0-.708.708l3 3z"/>
      </svg>
      <span>Download</span>
    `;
    btn.title = 'Download with VidSnap';
    
    btn.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      
      // Use current page URL if no specific video URL
      const url = videoUrl || window.location.href;
      const encodedUrl = encodeURIComponent(url);
      window.open(`${VIDSNAP_URL}/?url=${encodedUrl}&source=extension`, '_blank');
    });
    
    return btn;
  }
  
  // Find and process video elements
  function processVideos() {
    // Look for video containers on Facebook
    const videoContainers = document.querySelectorAll([
      '[data-pagelet*="Video"]',
      '[data-video-id]',
      'video',
      '[aria-label*="video"]'
    ].join(', '));
    
    videoContainers.forEach(container => {
      // Skip if already processed
      const videoId = container.getAttribute('data-video-id') || 
                      container.closest('[data-video-id]')?.getAttribute('data-video-id') ||
                      Math.random().toString(36).substr(2, 9);
      
      if (processedVideos.has(videoId)) return;
      processedVideos.add(videoId);
      
      // Find a good place to add the button
      const videoWrapper = container.closest('[role="article"]') || 
                           container.closest('[data-pagelet]') ||
                           container.parentElement;
      
      if (!videoWrapper || videoWrapper.querySelector('.vidsnap-download-btn')) return;
      
      // Try to get the video URL
      let videoUrl = window.location.href;
      const linkElement = videoWrapper.querySelector('a[href*="/watch"], a[href*="/videos/"], a[href*="/reel/"]');
      if (linkElement) {
        videoUrl = linkElement.href;
      }
      
      // Create and add the download button
      const btn = createDownloadButton(videoUrl);
      
      // Find action bar or create floating button
      const actionBar = videoWrapper.querySelector('[role="group"]');
      if (actionBar) {
        actionBar.appendChild(btn);
      } else {
        // Add as floating button
        btn.classList.add('vidsnap-floating');
        videoWrapper.style.position = 'relative';
        videoWrapper.appendChild(btn);
      }
    });
  }
  
  // Add floating download button when watching a video
  function addFloatingButton() {
    if (document.querySelector('.vidsnap-floating-main')) return;
    
    // Check if we're on a video page
    const isVideoPage = /\/watch|\/videos\/|\/reel\//.test(window.location.pathname);
    if (!isVideoPage) return;
    
    const btn = document.createElement('button');
    btn.className = 'vidsnap-floating-main';
    btn.innerHTML = `
      <svg width="20" height="20" fill="currentColor" viewBox="0 0 16 16">
        <path d="M.5 9.9a.5.5 0 0 1 .5.5v2.5a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-2.5a.5.5 0 0 1 1 0v2.5a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2v-2.5a.5.5 0 0 1 .5-.5z"/>
        <path d="M7.646 11.854a.5.5 0 0 0 .708 0l3-3a.5.5 0 0 0-.708-.708L8.5 10.293V1.5a.5.5 0 0 0-1 0v8.793L5.354 8.146a.5.5 0 1 0-.708.708l3 3z"/>
      </svg>
      Download with VidSnap
    `;
    btn.title = 'Download this video with VidSnap';
    
    btn.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      const encodedUrl = encodeURIComponent(window.location.href);
      window.open(`${VIDSNAP_URL}/?url=${encodedUrl}&source=extension`, '_blank');
    });
    
    document.body.appendChild(btn);
  }
  
  // Initialize
  function init() {
    // Initial processing
    setTimeout(() => {
      processVideos();
      addFloatingButton();
    }, 2000);
    
    // Watch for new content (Facebook loads content dynamically)
    const observer = new MutationObserver((mutations) => {
      let shouldProcess = false;
      mutations.forEach(mutation => {
        if (mutation.addedNodes.length > 0) {
          shouldProcess = true;
        }
      });
      if (shouldProcess) {
        setTimeout(processVideos, 500);
        addFloatingButton();
      }
    });
    
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
    
    // Also process on URL changes (SPA navigation)
    let lastUrl = location.href;
    new MutationObserver(() => {
      const url = location.href;
      if (url !== lastUrl) {
        lastUrl = url;
        processedVideos.clear();
        document.querySelectorAll('.vidsnap-download-btn, .vidsnap-floating-main').forEach(el => el.remove());
        setTimeout(() => {
          processVideos();
          addFloatingButton();
        }, 1000);
      }
    }).observe(document, { subtree: true, childList: true });
  }
  
  // Run when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
